<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\SalesReturn;
use App\SalesReturnItem;
use App\Stock;
use App\Tax;
use Validator;
use App\SaleTransaction;
use DB;
use Illuminate\Support\Facades\Auth;
use App\Contact;
use Illuminate\Validation\Rule;

class SaleTransactionController extends Controller
{

	/**
	 * Create a new controller instance.
	 *
	 * @return void
	 */
	public function __construct()
	{
		$this->middleware(function ($request, $next) {
			if (has_membership_system() == 'enabled') {
				if (!has_feature('inventory_module')) {
					if (!$request->ajax()) {
						return redirect('membership/extend')->with('message', _lang('Your Current package not support this feature. You can upgrade your package !'));
					} else {
						return response()->json(['result' => 'error', 'message' => _lang('Sorry, This feature is not available in your current subscription !')]);
					}
				}
			}

			return $next($request);
		});

		date_default_timezone_set(get_company_option('timezone', get_option('timezone', 'Asia/Dhaka')));
	}

	/**
	 * Display a listing of the resource.
	 *
	 * @return \Illuminate\Http\Response
	 */
	public function index()
	{
		$sales_returns = SalesReturn::where("company_id", company_id())
			->orderBy("id", "desc")->get();
		return view('backend.accounting.sales_return.list', compact('sales_returns'));
	}

	/**
	 * Show the form for creating a new resource.
	 *
	 * @return \Illuminate\Http\Response
	 */
	public function create(Request $request)
	{
		if (!$request->ajax()) {
			return view('backend.accounting.sales_return.create');
		} else {
			return view('backend.accounting.sales_return.modal.create');
		}
	}

	/**
	 * Store a newly created resource in storage.
	 *
	 * @param  \Illuminate\Http\Request  $request
	 * @return \Illuminate\Http\Response
	 */
	public function store(Request $request)
	{
		$validator = Validator::make($request->all(), [
			'return_date' => 'required',
			'customer_id' => 'required',
			'sub_total.*' => 'required|numeric',
			'attachemnt' => 'nullable|mimes:jpeg,png,jpg,doc,pdf,docx,zip',
		]);

		if ($validator->fails()) {
			if ($request->ajax()) {
				return response()->json(['result' => 'error', 'message' => $validator->errors()->all()]);
			} else {
				return redirect('sales_returns/create')
					->withErrors($validator)
					->withInput();
			}
		}

		DB::beginTransaction();

		$company_id = company_id();

		$attachemnt = "";
		if ($request->hasfile('attachemnt')) {
			$file = $request->file('attachemnt');
			$attachemnt = time() . $file->getClientOriginalName();
			$file->move(public_path() . "/uploads/attachments/", $attachemnt);
		}


		$salesReturn = new SalesReturn();
		$salesReturn->return_date = $request->input('return_date');
		$salesReturn->customer_id = $request->input('customer_id');

		if ($request->input('return_tax_id') != '') {
			$salesReturn->tax_id = $request->input('return_tax_id');
			$tax = Tax::find($salesReturn->tax_id);
			if ($tax->type == "percent") {
				$salesReturn->tax_amount = $request->input('product_total') * $tax->rate / (100 + $tax->rate);
			} else if ($tax->type == "fixed") {
				$salesReturn->tax_amount = $tax->rate;
			}
		}

		$salesReturn->product_total = $request->input('product_total');
		$salesReturn->grand_total = ($salesReturn->product_total + $salesReturn->tax_amount);
		$salesReturn->converted_total = convert_currency(base_currency(), $salesReturn->customer->currency, $salesReturn->grand_total);
		$salesReturn->attachemnt = $attachemnt;
		$salesReturn->note = $request->input('note');
		$salesReturn->company_id = $company_id;

		$salesReturn->save();


		//Save Sales Return item
		for ($i = 0; $i < count($request->product_id); $i++) {
			$salesReturnItem = new SalesReturnItem();
			$salesReturnItem->sales_return_id = $salesReturn->id;
			$salesReturnItem->product_id = $request->product_id[$i];
			$salesReturnItem->description = $request->product_description[$i];
			$salesReturnItem->quantity = $request->quantity[$i];
			$salesReturnItem->unit_cost = $request->unit_cost[$i];
			$salesReturnItem->discount = $request->discount[$i];
			$salesReturnItem->tax_method = $request->tax_method[$i];
			$salesReturnItem->tax_id = $request->tax_id[$i];
			$salesReturnItem->tax_amount = $request->tax_amount[$i];
			$salesReturnItem->sub_total = $request->sub_total[$i];
			$salesReturnItem->company_id = $company_id;
			$salesReturnItem->save();

			//Update Stock
			$stock = Stock::where("product_id", $salesReturnItem->product_id)
				->where("company_id", $company_id)->first();
			$stock->quantity = $stock->quantity + $salesReturnItem->quantity;
			$stock->company_id = $company_id;
			$stock->save();
		}

		DB::commit();


		if (!$request->ajax()) {
			return redirect('sales_returns/create')->with('success', _lang('Sales Returned Sucessfully'));
		} else {
			return response()->json(['result' => 'success', 'action' => 'store', 'message' => _lang('Sales Returned Sucessfully'), 'data' => $purchase]);
		}

	}


	/**
	 * Display the specified resource.
	 *
	 * @param  int  $id
	 * @return \Illuminate\Http\Response
	 */
	public function show(Request $request, $id)
	{
		$sales = SalesReturn::where("id", $id)->where("company_id", company_id())->first();
		if (!$request->ajax()) {
			return view('backend.accounting.sales_return.view', compact('sales', 'id'));
		} else {
			return view('backend.accounting.sales_return.modal.view', compact('sales', 'id'));
		}

	}

	/**
	 * Show the form for editing the specified resource.
	 *
	 * @param  int  $id
	 * @return \Illuminate\Http\Response
	 */
	public function edit(Request $request, $id)
	{
		$sales = SalesReturn::where("id", $id)->where("company_id", company_id())->first();
		if (!$request->ajax()) {
			return view('backend.accounting.sales_return.edit', compact('sales', 'id'));
		} else {
			return view('backend.accounting.sales_return.modal.edit', compact('sales', 'id'));
		}

	}

	/**
	 * Update the specified resource in storage.
	 *
	 * @param  \Illuminate\Http\Request  $request
	 * @param  int  $id
	 * @return \Illuminate\Http\Response
	 */
	public function update(Request $request, $id)
	{
		$validator = Validator::make($request->all(), [
			'return_date' => 'required',
			'customer_id' => 'required',
			'sub_total.*' => 'required|numeric',
			'attachemnt' => 'nullable|mimes:jpeg,png,jpg,doc,pdf,docx,zip',
		]);

		if ($validator->fails()) {
			if ($request->ajax()) {
				return response()->json(['result' => 'error', 'message' => $validator->errors()->all()]);
			} else {
				return redirect()->route('sales_returns.edit', $id)
					->withErrors($validator)
					->withInput();
			}
		}

		DB::beginTransaction();

		$company_id = company_id();

		$attachemnt = "";
		if ($request->hasfile('attachemnt')) {
			$file = $request->file('attachemnt');
			$attachemnt = time() . $file->getClientOriginalName();
			$file->move(public_path() . "/uploads/attachments/", $attachemnt);
		}


		$salesReturn = SalesReturn::where("id", $id)->where("company_id", $company_id)->first();
		$previous_amount = $salesReturn->grand_total;
		$salesReturn->return_date = $request->input('return_date');
		$salesReturn->customer_id = $request->input('customer_id');

		if ($request->input('return_tax_id') != "") {
			$salesReturn->tax_id = $request->input('return_tax_id');
			$tax = Tax::find($salesReturn->tax_id);
			if ($tax->type == "percent") {
				$salesReturn->tax_amount = $request->input('product_total') * $tax->rate / (100 + $tax->rate);
			} else if ($tax->type == "fixed") {
				$salesReturn->tax_amount = $tax->rate;
			}
		} else {
			$salesReturn->tax_id = null;
			$salesReturn->tax_amount = 0;
		}

		$salesReturn->product_total = $request->input('product_total');
		$salesReturn->grand_total = ($salesReturn->product_total + $salesReturn->tax_amount);
		if ($previous_amount != $salesReturn->grand_total) {
			$salesReturn->converted_total = convert_currency(base_currency(), $salesReturn->customer->currency, $salesReturn->grand_total);
		}
		$salesReturn->attachemnt = $attachemnt;
		$salesReturn->note = $request->input('note');
		$salesReturn->company_id = $company_id;

		$salesReturn->save();


		//Remove Previous Purcahse item
		$previous_items = SalesReturnItem::where("sales_return_id", $id)->get();
		foreach ($previous_items as $p_item) {
			$returnItem = SalesReturnItem::find($p_item->id);
			$returnItem->delete();
			$this->update_stock($p_item->product_id);
		}


		for ($i = 0; $i < count($request->product_id); $i++) {
			$returnItem = new SalesReturnItem();
			$returnItem->sales_return_id = $salesReturn->id;
			$returnItem->product_id = $request->product_id[$i];
			$returnItem->description = $request->product_description[$i];
			$returnItem->quantity = $request->quantity[$i];
			$returnItem->unit_cost = $request->unit_cost[$i];
			$returnItem->discount = $request->discount[$i];
			$returnItem->tax_method = $request->tax_method[$i];
			$returnItem->tax_id = $request->tax_id[$i];
			$returnItem->tax_amount = $request->tax_amount[$i];
			$returnItem->sub_total = $request->sub_total[$i];
			$returnItem->company_id = $company_id;
			$returnItem->save();

			$this->update_stock($request->product_id[$i]);

		}

		DB::commit();


		if (!$request->ajax()) {
			return redirect('sales_returns')->with('success', _lang('Updated Sucessfully'));
		} else {
			return response()->json(['result' => 'success', 'action' => 'update', 'message' => _lang('Updated Sucessfully'), 'data' => $purchase]);
		}

	}

	/**
	 * Remove the specified resource from storage.
	 *
	 * @param  int  $id
	 * @return \Illuminate\Http\Response
	 */
	public function destroy($id)
	{
		DB::beginTransaction();

		$salesReturn = SalesReturn::where("id", $id)
			->where("company_id", company_id());
		$salesReturn->delete();

		//Remove Sales Return Items
		$salesReturnItems = SalesReturnItem::where("sales_return_id", $id)->get();
		foreach ($salesReturnItems as $p_item) {
			$returnItem = SalesReturnItem::find($p_item->id);
			$returnItem->delete();
			$this->update_stock($p_item->product_id);
		}

		DB::commit();

		return redirect('sales_returns')->with('success', _lang('Deleted Sucessfully'));
	}


	private function update_stock($product_id)
	{
		$company_id = company_id();
		$purchase = DB::table('purchase_order_items')->where('product_id', $product_id)
			->where('company_id', $company_id)
			->sum('quantity');

		$purchaseReturn = DB::table('purchase_return_items')->where('product_id', $product_id)
			->where('company_id', $company_id)
			->sum('quantity');

		$sales = DB::table('invoice_items')->where('item_id', $product_id)
			->where('company_id', $company_id)
			->sum('quantity');

		$salesReturn = DB::table('sales_return_items')->where('product_id', $product_id)
			->where('company_id', $company_id)
			->sum('quantity');

		//Update Stock
		$stock = Stock::where("product_id", $product_id)->where("company_id", company_id())->first();
		$stock->quantity = ($purchase + $salesReturn) - ($sales + $purchaseReturn);
		$stock->save();
	}



	public function subscription_payment(Request $request, $id)
	{

		$datas = SaleTransaction::where("item_id", $id)->orderBy('id', 'desc')->get();
		$contact = Contact::where("id", $id)->first();
		//	 print_r($contact); die('xddd');

		return view('backend.accounting.sale_transcation.subscription_payment', compact('id', 'datas', 'contact'));

	}


	public function subscription_store(Request $request, $id)
	{

		SaleTransaction::create(array_merge($request->all(), ['item_id' => $id]));

		//return view('backend.accounting.sale_transcation.subscription_payment_list',compact('id'));

		return redirect()->back();

		//return $request->all();

	}


	// customer reports 




	public function customer_reports22(Request $request)
	{
		$company_id = company_id();
		$contact_from_date = $request->contact_from_date;
		$contact_to_date = $request->contact_to_date;
		// $assign_lead_id=$request->assign_lead_id??'0';
		$transaction_from_date = $request->transaction_from_date;
		$transaction_to_date = $request->transaction_to_date;
		$paginate = $request->paginate;

		$datas = Contact::with([
			'transactions' => function ($q) use ($transaction_from_date, $transaction_to_date) {
				$q->whereDate('payment_date', '>=', $transaction_from_date)
					->whereDate('payment_date', '<=', $transaction_to_date);
			},
			'transactions.group',
			'assignuser'
		])->orderBy('id', 'desc');

		if (!empty($contact_from_date)) {
			$datas = $datas->whereDate('created_at', '>=', $contact_from_date);
		}

		if (!empty($contact_to_date)) {
			$datas = $datas->whereDate('created_at', '<=', $contact_to_date);
		}

		if ($request->assigned_user_id) {
			$datas = $datas->where('converted_by_user_id', $request->assigned_user_id);

		} else {
			if (Auth::user()->role_id == 8) {
				$datas = $datas->where('converted_by_user_id', Auth::user()->id);
			}
		}

		if ($request->time_zone) {
			$datas = $datas->where('time_zone', $request->time_zone);
		}
		if ($request->zone_state) {
			$datas = $datas->where('zone_state', $request->zone_state);
		}


		if (!empty($transaction_from_date) && !empty($transaction_to_date)) {
			$datas = $datas->whereHas('transactions', function ($q) use ($transaction_from_date, $transaction_to_date) {
				$q->whereDate('payment_date', '>=', $transaction_from_date)
					->whereDate('payment_date', '<=', $transaction_to_date);
			});

		}
		if ($paginate == 1) {
			$datas = $datas->where('company_id', $company_id)->get();
		} else {
			$datas = $datas->where('company_id', $company_id)->paginate(10);
		}
		// return $datas;
		// die;
		$refundData = [];
		return view('backend.accounting.sale_transcation.customer_reports_list', compact('refundData', 'datas', 'paginate'));


	}



	// ---------------------------------------------------- 
	public function customer_reports(Request $request)
	{

	
		$company_id = company_id();
		$user = Auth::user();
		$paginate = $request->paginate;


		// Shared base query logic
		$buildBaseQuery = function () use ($request, $user, $company_id) {

			$with = [
				'transactions.group',
				'assignuser'
			];

			// Conditionally filter eager-loaded transactions
			if (!empty($request->transaction_from_date) && !empty($request->transaction_to_date)) {
				$fromDate = $request->transaction_from_date;
				$toDate = $request->transaction_to_date;

				$with['transactions'] = function ($q) use ($fromDate, $toDate) {
					$q->whereDate('payment_date', '>=', $fromDate)
						->whereDate('payment_date', '<=', $toDate);
				};
			}

			$query = Contact::with($with)
				->where('company_id', $company_id)
				->orderByDesc('id');

			if (!empty($request->contact_from_date)) {
				$query->whereDate('created_at', '>=', $request->contact_from_date);
			}

			if (!empty($request->contact_to_date)) {
				$query->whereDate('created_at', '<=', $request->contact_to_date);
			}

			if (!empty($request->assigned_user_id)) {
				$query->where('converted_by_user_id', $request->assigned_user_id);
			} elseif ($user->role_id == 8) {
				$query->where('converted_by_user_id', $user->id);
			}

			if (!empty($request->time_zone)) {
				$query->where('time_zone', $request->time_zone);
			}

			if (!empty($request->zone_state)) {
				$query->where('zone_state', $request->zone_state);
			}

			return $query;
		};

		// Main transaction data
		$datasQuery = $buildBaseQuery();
		if (!empty($request->transaction_from_date) && !empty($request->transaction_to_date)) {
			$datasQuery->whereHas('transactions', function ($q) use ($request) {
				$q->whereDate('payment_date', '>=', $request->transaction_from_date)
					->whereDate('payment_date', '<=', $request->transaction_to_date);
			});
		}
		 
		$datas = $paginate == 1 ? $datasQuery->get() : $datasQuery->paginate(10);
 

		$buildRefundQuery = function () use ($request, $user, $company_id) {
			$with = [
				'transactions.group',
				'assignuser'
			];

			// Conditionally filter eager-loaded transactions
			if (!empty($request->transaction_from_date) && !empty($request->transaction_to_date)) {
				$fromDate = $request->transaction_from_date;
				$toDate = $request->transaction_to_date;

				$with['transactions'] = function ($q) use ($fromDate, $toDate) {
					$q->whereDate('refund_payment_date', '>=', $fromDate)
						->whereDate('refund_payment_date', '<=', $toDate)
						->whereNotNull('refund_payment_date');
				};
			}

			$query = Contact::with($with)
				->where('company_id', $company_id)
				->orderByDesc('id');

			if (!empty($request->contact_from_date)) {
				$query->whereDate('created_at', '>=', $request->contact_from_date);
			}

			if (!empty($request->contact_to_date)) {
				$query->whereDate('created_at', '<=', $request->contact_to_date);
			}

			if (!empty($request->assigned_user_id)) {
				$query->where('converted_by_user_id', $request->assigned_user_id);
			} elseif ($user->role_id == 8) {
				$query->where('converted_by_user_id', $user->id);
			}

			if (!empty($request->time_zone)) {
				$query->where('time_zone', $request->time_zone);
			}

			if (!empty($request->zone_state)) {
				$query->where('zone_state', $request->zone_state);
			}

			return $query;
		};

		// 👉 Now call the closure to get the query builder
		$refundQuery = $buildRefundQuery();

		// Apply whereHas to filter contacts with valid refund transactions
		if (!empty($request->transaction_from_date) && !empty($request->transaction_to_date)) {
			$fromDate = $request->transaction_from_date;
			$toDate = $request->transaction_to_date;

			$refundQuery->whereHas('transactions', function ($q) use ($fromDate, $toDate) {
				$q->whereDate('refund_payment_date', '>=', $fromDate)
					->whereDate('refund_payment_date', '<=', $toDate)
					->whereNotNull('refund_payment_date');
			});
		}

		// Fetch the data
	// DB::enableQueryLog();

	$refundData = $paginate == 1 ? $refundQuery->get() : $refundQuery->paginate(10);
	// dd(DB::getQueryLog());

		// Remove overlapping contacts (based on ID)
		if ($paginate == 1) {
			$refundData = $refundData->reject(function ($refund) use ($datas) {
				return $datas->contains('id', $refund->id);
			});
		} else {
			// If paginated, convert to collection to filter, then re-paginate manually
			$datasIds = $datas->pluck('id')->toArray();
			$filteredRefund = $refundData->getCollection()->reject(function ($refund) use ($datasIds) {
				return in_array($refund->id, $datasIds);
			});

			// Rebuild paginator with filtered data
			$refundData = new \Illuminate\Pagination\LengthAwarePaginator(
				$filteredRefund,
				$filteredRefund->count(),
				$refundData->perPage(),
				$refundData->currentPage(),
				['path' => request()->url(), 'query' => request()->query()]
			);
		}


//   $taaa = $refundData->flatMap->transactions->sum('refundable_amount');
//             echo $taaa;
//             die;

		return view('backend.accounting.sale_transcation.customer_reports_list', compact('datas', 'refundData', 'paginate'));
	}




	// Refund

	public function customer_refund(Request $request, $id)
	{
		$SaleTransaction = SaleTransaction::find($id);

		return view('backend.accounting.sale_transcation.customer_refunds', compact('id', 'SaleTransaction'));

	}




	public function refund_status_store(Request $request, $id)
	{

		//  dd($id);
		// dd($request->all());

		$SaleTransaction = SaleTransaction::find($id);
		if ($request->refundable_amount <= $SaleTransaction->amount) {
			$SaleTransaction->update($request->all());
			return redirect()->back()->with('success', _lang('Refund Updated Sucessfully'));
		} else {
			return redirect()->back()->with('error', _lang('Refund Amount Greater then paid amount '));
		}

	}




	public function refund_status_dispute(Request $request, $id)
	{

		$dispute = SaleTransaction::find($id);
		if ($dispute) {
			$dispute->update(['refund_type' => 'Dispute', 'refundable_amount' => $dispute->amount ?? 0, 'refund_payment_date' => date("Y-m-d H:i:s")]);
		}
		return redirect()->back()->with('success', _lang('Dispute Sucessfully'));
	}

	public function transactionedit(Request $request, $id)
	{
		$data = SaleTransaction::find($id);
		$contact = Contact::where("id", $data->item_id)->first();
		return view('backend.accounting.sale_transcation.edit', compact('data', 'contact'));

	}
	public function transactionupdate(Request $request, $id)
	{
		$data = SaleTransaction::find($id);
		$data->update($request->all());
		return redirect()->back()->with('success', _lang('Update Successfully'));

	}
	public function saletransactiondelete(Request $request, $id)
	{
		$data = SaleTransaction::find($id);
		if ($data) {
			$data->delete();
			return redirect()->back()->with('success', _lang('Deleted Successfully'));
		}

		return redirect()->back()->with('error', _lang('Data Not Deleted'));
	}
}
